import 'package:flutter/material.dart';
import 'package:flutter_form_builder/flutter_form_builder.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:gap/gap.dart';
import 'package:razinsoft_rider/core/routes/app_routes.dart';
import 'package:razinsoft_rider/core/utils/exit_app_dialogue.dart';
import 'package:razinsoft_rider/core/utils/get_version.dart';
import 'package:razinsoft_rider/core/utils/helpers.dart';
import 'package:razinsoft_rider/data/services/navigation_service.dart';
import 'package:razinsoft_rider/presentation/account_page/provider/theme_provider.dart';
import 'package:razinsoft_rider/presentation/auth/widgets/auth_app_bar.dart';
import 'package:razinsoft_rider/presentation/auth/widgets/auth_bottom_buttons.dart';
import 'package:razinsoft_rider/presentation/auth/widgets/auth_message.dart';

import '../../../core/widgets/custom_phone_field.dart';
import '../../../data/models/login_response.dart';
import '../../../generated/l10n.dart';
import '../../account_page/provider/select_country_provider.dart';
import '../provider/auth_providers.dart';

class LoginPage extends ConsumerStatefulWidget {
  const LoginPage({super.key});

  @override
  ConsumerState<LoginPage> createState() => _LoginPageState();
}

class _LoginPageState extends ConsumerState<LoginPage> {
  final formKey = GlobalKey<FormBuilderState>();

  bool isPhoneValidLength = false;

  String phoneNumber = '';
  String version = '';

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((_){
      Future.microtask(()=> loadVersion());
    });
  }

  Future<void> loadVersion()async{
    version = await getVersion();
    setState(() {

    });
  }
  @override
  Widget build(BuildContext context) => ExitAppWrapper(
      child: Scaffold(
        resizeToAvoidBottomInset: true,
        body: AuthAppBar(
          showLeading: false,
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              authMessage(context, isDarkMode: ref.read(themeModeProvider.notifier).isDarkMode(), version: version),
              Gap(12.h),
              FormBuilder(
                key: formKey,
                child: AppPhoneNumberTextField(
                  initialValue: '',
                  onChanged: (value) {
                    if (value != null) {
                      phoneNumber = value;
                      setState(() {
                        isPhoneValidLength = value.length >= 6;
                      });
                    }
                  },
                ),
              ),

            ],
          ),
        ),

        bottomNavigationBar: Consumer(
          builder: (context, ref, _) {
            final state = ref.watch(loginNotifierProvider);
            final stateNotifier = ref.read(loginNotifierProvider.notifier);

            // handle navigation
            return AuthBottomButtons(
                isLoading: state.when(initial: ()=> false, loading: ()=> true, success: (data)=> false, error: (e)=> false),
                onSkip: () {
                  NavigationService.pushNamedAndRemoveUntil(AppRoutes.dashboard,);
                },
                title: AppLocalizations().loginSignup,
                onTap: () {
                  if(phoneNumber.trim().length < 6){
                    showNotification( message: AppLocalizations().phoneMinLengthError);
                    return;
                  }
                  final String? phoneCode = ref.read(selectedCountry).selectedPhoneCode?.phoneCode;
                  if(phoneCode == null){
                    showNotification(message: 'Phone code not found');
                    return;
                  }
                  final String phone = phoneNumber;
                  stateNotifier.login(phone: phone, countryCode: phoneCode);
                });
          },
        ),
      ),
    );

  bool isNewRider(LoginResponse response) => response.data?.isNewRider ?? false;
}

